<?php

/* Stemmer for Serbian language, based on:
 *
 * Ljubesic, Nikola. Pandzic, Ivan. Stemmer for Croatian
 * http://nlp.ffzg.hr/resources/tools/stemmer-for-croatian/
 *
 * authors: Stefan Petkovic and Dragan Ivanovic
 * emails: petkovic8 at gmail.com and dragan.ivanovic at uns.ac.rs
 * version: 1.0 (20.04.2019)
*/
class Wpil_Stemmer {
    static $no_diacritics;
    static $p1;

    /* special characters - Unicode codepoints */

    /* serbian cyrillic */
    static $cyr_alphabet = array(
        'cyrA' => 'а',
        'cyrB' => 'б',
        'cyrV' => 'в',
        'cyrG' => 'г',
        'cyrD' => 'д',
        'cyrDx' => 'ђ',
        'cyrE' => 'е',
        'cyrZh' => 'ж',
        'cyrZ' => 'з',
        'cyrI' => 'и',
        'cyrJ' => 'ј',
        'cyrK' => 'к',
        'cyrL' => 'л',
        'cyrLJ' => 'љ',
        'cyrM' => 'м',
        'cyrN' => 'н',
        'cyrNJ' => 'њ',
        'cyrO' => 'о',
        'cyrP' => 'п',
        'cyrR' => 'р',
        'cyrS' => 'с',
        'cyrT' => 'т',
        'cyrCy' => 'ћ',
        'cyrU' => 'у',
        'cyrF' => 'ф',
        'cyrH' => 'х',
        'cyrC' => 'ц',
        'cyrCx' => 'ч',
        'cyrDzx' => 'џ',
        'cyrSx' => 'ш',
    );

    /* serbian latin with diacritics */
    static $cx = 'č'; // small c with caron
    static $cy = 'ć'; // small c with acute
    static $zx = 'ž'; // small z with caron
    static $sx = 'š'; // small s with caron
    static $dx = 'đ'; // small d with stroke

    static $vowels = array(
        'a',
        'e',
        'i',
        'o',
        'u'
    );

    static $sa = array(
        'č',
        'ć',
        'ž',
        'š',
        'đ'
    );

    /* consonant list */
    static $ca = 'bvgdzjklmnprstfhcčćžšđ';
    static $rg = 'r';

    public static function cyr_to_lat($word){

        $cyr_replace = array(
            'cyrA' => 'a',
            'cyrB' => 'b',
            'cyrV' => 'v',
            'cyrG' => 'g',
            'cyrD' => 'd',
            'cyrDx'=> self::$dx,
            'cyrE' => 'e',
            'cyrZh'=> self::$zx,
            'cyrZ' => 'z',
            'cyrI' => 'i',
            'cyrJ' => 'j',
            'cyrK' => 'k',
            'cyrL' => 'l',
            'cyrLJ'=> 'lj',
            'cyrM' => 'm',
            'cyrN' => 'n',
            'cyrNJ'=> 'nj',
            'cyrO' => 'o',
            'cyrP' => 'p',
            'cyrR' => 'r',
            'cyrS' => 's',
            'cyrT' => 't',
            'cyrCy'=> self::$cy,
            'cyrU' => 'u',
            'cyrF' => 'f',
            'cyrH' => 'h',
            'cyrC' => 'c',
            'cyrCx'=> self::$cx,
            'cyrDzx'=> 'd' . self::$zx,
            'cyrSx'=> self::$sx,
        );

        $new_word = '';
        $cyr_letters = array_flip(self::$cyr_alphabet);
        $strLen = mb_strlen($word, 'UTF-8');
        for ($i = 0; $i < $strLen; $i++)
        {
            $letter = mb_substr($word, $i, 1, 'UTF-8');
            $cyr_letter = isset($cyr_letters[$letter]) ? $cyr_letters[$letter]: false;
            if($cyr_letter){
                $letter = $cyr_replace[$cyr_letter];
            }

            $new_word .= $letter;
        }

        return $new_word;
    }
    
    public static function prelude ($word){

        $len = mb_strlen($word);
        $pos = 0;

        while(true){
            // check the word for 'ije'
            $pos = mb_strpos($word, 'ije', $pos);

            // if it's not present, break out of the loop
            if($pos === false){
                break;
            }

            // get the letters before and after 'ije'
            $bef = mb_substr($word, $pos - 1, 1);
            $aft = mb_substr($word, $pos + 3, 1);

            // if the 'ije' text falls between 2 listed consonants
            if( !empty($bef) && false !== mb_stripos(self::$ca, $bef) && 
                !empty($aft) && false !== mb_stripos(self::$ca, $aft)){
                // replace the 'ije' with 'e'
                $word_a = mb_substr($word, 0, $pos);
                $word_b = mb_substr($word, $pos + 3);
                $word = $word_a . 'e' . $word_b;
            }
            $pos++;
        }

        // reset the position
        $pos = 0;

        while(true){
            // check the word for 'je'
            $pos = mb_strpos($word, 'je', $pos);

            // if it's not present, break out of the loop
            if($pos === false){
                break;
            }

            // get the letters before and after 'je'
            $bef = mb_substr($word, $pos - 1, 1);
            $aft = mb_substr($word, $pos + 2, 1);

            // if the 'je' text falls between 2 listed consonants
            if( !empty($bef) && false !== mb_stripos(self::$ca, $bef) && 
                !empty($aft) && false !== mb_stripos(self::$ca, $aft)){
                // replace the 'je' with 'e'
                $word_a = mb_substr($word, 0, $pos);
                $word_b = mb_substr($word, $pos + 2);
                $word = $word_a . 'e' . $word_b;
            }
            // increment the position to advance
            $pos++;
        }

        // replace any instances of 'dj' with 'đ'
        $word = mb_ereg_replace('dj', self::$dx, $word);

        return $word;
    }

    public static function mark_regions($word){

        self::$no_diacritics = true;

        // see if the word contains diacritics
        foreach(self::$sa as $a){
            // if it does
            if(false !== mb_strpos($word, $a)){
                // unset the no_diacritics flag to change word handling
                self::$no_diacritics = false;
                break;
            }
        }

        // set p1 as the limit (strlen)
        self::$p1 = mb_strlen($word);

        // go past the first vowel
        // set $p1 for the location
        // if $p1 is less than 2
        // go past the next non vowel and set $p1 to that position

        // find the first vowel in the word
        $shortest = false;
        foreach(self::$vowels as $vowel){
            $pos = mb_strpos($word, $vowel);
            if($pos !== false && ($shortest === false || $pos < $shortest)){
                $shortest = $pos;
            }
        }

        // if the sentence contains a vowel
        if($shortest !== false){
            // set p1 to the vowel's position
            self::$p1 = $shortest;

            // if the vowel doesn't have 2 consonants before it, set p1 to the first consonant after the vowel
            if(self::$p1 < 2){
                $len = mb_strlen($word);
                for($i = 0; $i < $len; $i++){
                    if($i > self::$p1){
                        $char = mb_substr($word, $i, 1);
                        if(!in_array($char, self::$vowels, true)){
                            self::$p1 = $i;
                        }
                    }
                }
            }
        }

        // get the position after the first r
        // if the position is greater than or equal to 2 || there's a non rg char after the pos
        // test if p1 minus cursor is greater than 1
        // if it is, set p1 to the pos
        $past_r = mb_stripos($word, 'r');
        if(false !== $past_r){
            $char = mb_substr($word, $past_r + 1, 1);
            if($past_r >= 2){
                // if the 'r' is closer to the start of the sentence than p1, set p1 for the 'r'
                if(self::$p1 - $past_r > 1){
                    self::$p1 = $past_r;
                }
            }elseif($char !== 'r'){
                // if the char after the first 'r' is closer to the start of the sentence than p1, set p1 for the char's position
                if(self::$p1 - $past_r > 1){
                    self::$p1 = $past_r + 1;
                }
            }
        }
    }

    public static function backwards_step_1($word){
        // get the word length
        $length = mb_strlen($word);

        // created the list of word endings to update
        $strings = array(
            'lozi'      => 'loga',
            'lozima'    => 'loga',
            'pesi'      => 'peh',
            'pesima'    => 'peh',
            'vojci'     => 'vojka',
            'bojci'     => 'bojka',
            'jaci'      => 'jak',
            'jacima'    => 'jak',
            'čajan'     => 'čajni',
            'cajan'     => array('no_diacritics' => 'cajni'),
            'eran'      => 'erni',
            'laran'     => 'larni',
            'esan'      => 'esni',
            'anjac'     => 'anjca',
            'ajac'      => 'ajca',
            'ajaca'     => 'ajca',
            'ljaca'     => 'ljca',
            'ljac'      => 'ljca',
            'ejac'      => 'ejca',
            'ejaca'     => 'ejca',
            'ojac'      => 'ojca',
            'ojaca'     => 'ojca',
            'ajaka'     => 'ajka',
            'ojaka'     => 'ojka',
            'šaca'      => 'šca',
            'šac'       => 'šca',
            'inzima'    => 'ing',
            'inzi'      => 'ing',
            'tvenici'   => 'tvenik',
            'tetici'    => 'tetika',
            'teticima'  => 'tetika',
            'nstava'    => 'nstva',
            'nicima'    => 'nik',
            'ticima'    => 'tik',
            'zicima'    => 'zik',
            'snici'     => 'snik',
            'kuse'      => 'kusi',
            'kusan'     => 'kusni',
            'kustava'   => 'kustva',
            'dušan'     => 'dušni',
            'dusan'     => array('no_diacritics' => 'dusni'),
            'antan'     => 'antni',
            'bilan'     => 'bilni',
            'tilan'     => 'tilni',
            'avilan'    => 'avilni',
            'silan'     => 'silni',
            'gilan'     => 'gilni',
            'rilan'     => 'rilni',
            'nilan'     => 'nilni',
            'alan'      => 'alni',
            'ozan'      => 'ozni',
            'rave'      => 'ravi',
            'stavan'    => 'stavni',
            'pravan'    => 'pravni',
            'tivan'     => 'tivni',
            'sivan'     => 'sivni',
            'atan'      => 'atni',
            'enat'      => 'enta',
            'tetan'     => 'tetni',
            'pletan'    => 'pletni',
            'šave'      => 'šavi',
            'save'      => array('no_diacritics' =>'savi'),
            'anata'     => 'anta',
            'ačak'      => 'ačka',
            'ačaka'     => 'ačka',
            'acak'      => array('no_diacritics' =>'acka'),
            'acaka'     => array('no_diacritics' =>'acka'),
            'ušak'      => 'uška',
            'usak'      => array('no_diacritics' =>'uska'),
            'atak'      => 'atka',
            'ataka'     => 'atka',
            'atci'      => 'atka',
            'atcima'    => 'atka',
            'etak'      => 'etka',
            'etaka'     => 'etka',
            'itak'      => 'itka',
            'itaka'     => 'itka',
            'itci'      => 'itka',
            'otak'      => 'otka',
            'otaka'     => 'otka',
            'utak'      => 'utka',
            'utaka'     => 'utka',
            'utci'      => 'utka',
            'utcima'    => 'utka',
            'eskan'     => 'eskna',
            'tičan'     => 'tični',
            'tican'     => array('no_diacritics' =>'ticni'),
            'ojsci'     => 'ojska',
            'esama'     => 'esma',
            'metar'     => 'metra',
            'metara'    => 'metra',
            'centar'    => 'centra',
            'centara'   => 'centra',
            'istar'     => 'istra',
            'istara'    => 'istra',
            'ošću'      => 'osti',
            'oscu'      => array('no_diacritics' =>'osti'),
            'daba'      => 'dba',
            'čcima'     => 'čka',
            'čci'       => 'čka',
            'mac'       => 'mca',
            'maca'      => 'mca',
            'naca'      => 'nca',
            'nac'       => 'nca',
            'voljan'    => 'voljni',
            'anaka'     => 'anki',
            'vac'       => 'vca',
            'vaca'      => 'vca',
            'saca'      => 'sca',
            'sac'       => 'sca',
            'raca'      => 'rca',
            'rac'       => 'rca',
            'aoca'      => 'alca',
            'alaca'     => 'alca',
            'alac'      => 'alca',
            'elaca'     => 'elca',
            'elac'      => 'elca',
            'olaca'     => 'olca',
            'olac'      => 'olca',
            'olce'      => 'olca',
            'njac'      => 'njca',
            'njaca'     => 'njca',
            'ekata'     => 'ekta',
            'ekat'      => 'ekta',
            'izam'      => 'izma',
            'izama'     => 'izma',
            'jebe'      => 'jebi',
            'baci'      => 'baci',
            'ašan'      => 'ašni',
            'asan'      => array('no_diacritics' =>'asni'),
        );

        foreach($strings as $s_key => $s_replace){
            // if this is a string replace and the string key is in the word
            if(is_string($s_replace) && false !== strpos($word, $s_key)){
                $key_len = mb_strlen($s_key);
                $key_pos = mb_strrpos($word, $s_key);

                // see if the key comes at the end of the word by adding the key pos and it's length and seeing if that equals the word length
                if(($key_len + $key_pos) === $length){
                    // if it does, change the ending
                    $word = mb_substr($word, 0, $key_pos) . $s_replace;
                    // and exit the loop
                    break;
                }

            }elseif(self::$no_diacritics && false !== strpos($word, $s_key)){
                $key_len = mb_strlen($s_key);
                $key_pos = mb_strrpos($word, $s_key);

                // see if the key comes at the end of the word by adding the key pos and it's length and seeing if that equals the word length
                if(($key_len + $key_pos) === $length){
                    // if it does, change the ending
                    $word = mb_substr($word, 0, $key_pos) . $s_replace['no_diacritics'];
                    // and exit the loop
                    break;
                }
            }
        }

        return $word;
    }

    public static function backwards_step_2(&$word){

        $r1 = !empty(self::$p1) ? self::$p1: false;

        if(empty($r1)){ // todo I _think_ this is correct, test to make sure
            return false;
        }

        // get the word length
        $length = mb_strlen($word);

        // create the list of word endings to update
        $strings = array (
            'ovitijega' => 'ovit',
            'astajasmo' => 'astaj',
            'astadosmo' => 'asta',
            'astadoste' => 'asta',
            'ostajaste' => 'ostaj',
            'ostajasmo' => 'ostaj',
            'istadosmo' => 'ista',
            'istadoste' => 'ista',
            'istajaste' => 'istaj',
            'istajasmo' => 'istaj',
            'ovitijima' => 'ovit',
            'ostadosmo' => 'osta',
            'astajaste' => 'astaj',
            'ostadoste' => 'osta',
            'evitijemu' => 'evit',
            'ovitijemu' => 'ovit',
            'evitijega' => 'evit',
            'evitijima' => 'evit',
            'astanimo' => 'asta',
            'lajevima' => 'laj',
            'istadoše' => 'ista',
            'đajevima' => 'đaj',
            'ćajevima' => 'ćaj',
            'astanila' => 'asta',
            'astanite' => 'asta',
            'lucujemo' => array('no_diacritics' => 'luc'),
            'istanemo' => 'ista',
            'astanete' => 'asta',
            'astademo' => 'asta',
            'astanemo' => 'asta',
            'snjijima' => array('no_diacritics' => 'snj'),
            'snjijemu' => array('no_diacritics' => 'snj'),
            'astadoše' => 'asta',
            'snjijega' => array('no_diacritics' => 'snj'),
            'ostajale' => 'ostaj',
            'astadete' => 'asta',
            'istanete' => 'ista',
            'istademo' => 'ista',
            'ostademo' => 'osta',
            'lučujemo' => 'luč',
            'lučujući' => 'luč',
            'lučujete' => 'luč',
            'ostanila' => 'osta',
            'ostanite' => 'osta',
            'ostanimo' => 'osta',
            'ostadete' => 'osta',
            'ostanete' => 'osta',
            'ostanemo' => 'osta',
            'lucujuci' => array('no_diacritics' => 'luc'),
            'ostadoše' => 'osta',
            'lucujete' => array('no_diacritics' => 'luc'),
            'rajevima' => 'raj',
            'ostajala' => 'ostaj',
            'istanila' => 'ista',
            'istanite' => 'ista',
            'istanimo' => 'ista',
            'istadete' => 'ista',
            'ostajalo' => 'ostaj',
            'ostajati' => 'ostaj',
            'ostajući' => 'ostaj',
            'astijemu' => 'ast',
            'anjijemu' => 'anj',
            'anjijima' => 'anj',
            'anjijega' => 'anj',
            'enjijemu' => 'enj',
            'enjijima' => 'enj',
            'enjijega' => 'enj',
            'šnjijemu' => 'šnj',
            'šnjijima' => 'šnj',
            'šnjijega' => 'šnj',
            'astijega' => 'ast',
            'astajati' => 'astaj',
            'astijima' => 'ast',
            'ovitijeg' => 'ovit',
            'ovitijoj' => 'ovit',
            'ovitijih' => 'ovit',
            'evitijem' => 'evit',
            'evitijim' => 'evit',
            'evitijih' => 'evit',
            'evitijoj' => 'evit',
            'evitijeg' => 'evit',
            'ovitijem' => 'ovit',
            'astajahu' => 'astaj',
            'astajemo' => 'astaj',
            'ostajali' => 'ostaj',
            'istajali' => 'istaj',
            'ostajaše' => 'ostaj',
            'ostajete' => 'ostaj',
            'ostajemo' => 'ostaj',
            'cajevima' => 'caj',
            'ostajahu' => 'ostaj',
            'istajale' => 'istaj',
            'istajalo' => 'istaj',
            'istajala' => 'istaj',
            'istajući' => 'istaj',
            'istajaše' => 'istaj',
            'astajete' => 'astaj',
            'istajete' => 'istaj',
            'istajemo' => 'istaj',
            'istajati' => 'istaj',
            'istajahu' => 'istaj',
            'astajale' => 'astaj',
            'astajalo' => 'astaj',
            'astajala' => 'astaj',
            'astajući' => 'astaj',
            'astajali' => 'astaj',
            'astajaše' => 'astaj',
            'čajevima' => 'čaj',
            'ovitijim' => 'ovit',
            'tanijama' => 'tanij',
            'astajase' => array('no_diacritics' => 'astaj'),
            'ganijama' => 'ganij',
            'ostajuci' => array('no_diacritics' => 'ostaj'),
            'ostajase' => array('no_diacritics' => 'ostaj'),
            'tanijima' => 'tanij',
            'istajuci' => array('no_diacritics' => 'istaj'),
            'astadose' => array('no_diacritics' => 'asta'),
            'ganijima' => 'ganij',
            'manijima' => 'manij',
            'istajase' => array('no_diacritics' => 'istaj'),
            'istadose' => array('no_diacritics' => 'ista'),
            'astajuci' => array('no_diacritics' => 'astaj'),
            'manijama' => 'manij',
            'ostadose' => array('no_diacritics' => 'osta'),
            'panijama' => 'panij',
            'ranijima' => 'ranij',
            'ranijama' => 'ranij',
            'panijima' => 'panij',
            'evajuci' => array('no_diacritics' => 'eva'),
            'ivajuci' => array('no_diacritics' => 'iva'),
            'avijima' => 'av',
            'avijega' => 'av',
            'urajući' => 'ur',
            'astajat' => 'astaj',
            'avajuci' => array('no_diacritics' => 'ava'),
            'evijemu' => 'ev',
            'škijima' => 'šk',
            'ostavsi' => array('no_diacritics' => 'osta'),
            'evijima' => 'ev',
            'avijemu' => 'av',
            'atijima' => 'at',
            'škijega' => 'šk',
            'škijemu' => 'šk',
            'astaješ' => 'astaj',
            'uvajuci' => array('no_diacritics' => 'uva'),
            'atijemu' => 'at',
            'astajte' => 'astaj',
            'atijega' => 'at',
            'astajmo' => 'astaj',
            'astajao' => 'astaj',
            'astajem' => 'astaj',
            'astajah' => 'astaj',
            'evijega' => 'ev',
            'lucismo' => array('no_diacritics' => 'luc'),
            'lucujem' => array('no_diacritics' => 'luc'),
            'uvajući' => 'uva',
            'anjijim' => 'anj',
            'anjijem' => 'anj',
            'anjijeg' => 'anj',
            'ivajući' => 'iva',
            'snjijoj' => array('no_diacritics' => 'snj'),
            'snjijeg' => array('no_diacritics' => 'snj'),
            'snjijih' => array('no_diacritics' => 'snj'),
            'snjijim' => array('no_diacritics' => 'snj'),
            'elijega' => 'el',
            'elijima' => 'el',
            'astajes' => array('no_diacritics' => 'astaj'),
            'elijemu' => 'el',
            'irujemo' => 'ir',
            'lucujes' => array('no_diacritics' => 'luc'),
            'irujete' => 'ir',
            'irujući' => 'ir',
            'irajući' => 'ir',
            'ovijega' => 'ov',
            'ovijima' => 'ov',
            'ovijemu' => 'ov',
            'ivijega' => 'iv',
            'ivijima' => 'iv',
            'ivijemu' => 'iv',
            'luciste' => array('no_diacritics' => 'luc'),
            'jetiste' => 'jeti',
            'lucujte' => array('no_diacritics' => 'luc'),
            'lucujmo' => array('no_diacritics' => 'luc'),
            'ostanes' => array('no_diacritics' => 'osta'),
            'istajah' => 'istaj',
            'jetismo' => 'jeti',
            'manijom' => 'manij',
            'istajes' => array('no_diacritics' => 'istaj'),
            'tanijom' => 'tanij',
            'ozilima' => 'ozil',
            'olovima' => 'olov',
            'astavši' => 'asta',
            'astanem' => 'asta',
            'astadem' => 'asta',
            'astaneš' => 'asta',
            'astadeš' => 'asta',
            'astadoh' => 'asta',
            'injaste' => 'inj',
            'injasmo' => 'inj',
            'ostadoh' => 'osta',
            'ijalima' => 'ijal',
            'ostadeš' => 'osta',
            'ostaneš' => 'osta',
            'ostadem' => 'osta',
            'ostanem' => 'osta',
            'ostavši' => 'osta',
            'ranijom' => 'ranij',
            'panijom' => 'panij',
            'istavši' => 'ista',
            'istanem' => 'ista',
            'istadem' => 'ista',
            'istaneš' => 'ista',
            'istadeš' => 'ista',
            'skijima' => 'sk',
            'ijalama' => 'ijal',
            'ajevima' => 'aj',
            'ostades' => array('no_diacritics' => 'osta'),
            'ikatima' => 'ikat',
            'istajmo' => 'istaj',
            'istajao' => 'istaj',
            'istajem' => 'istaj',
            'istaješ' => 'istaj',
            'istajat' => 'istaj',
            'anjijoj' => 'anj',
            'istajte' => 'istaj',
            'istavsi' => array('no_diacritics' => 'ista'),
            'istanes' => array('no_diacritics' => 'ista'),
            'istades' => array('no_diacritics' => 'ista'),
            'astavsi' => array('no_diacritics' => 'asta'),
            'astanes' => array('no_diacritics' => 'asta'),
            'astades' => array('no_diacritics' => 'asta'),
            'ostajes' => array('no_diacritics' => 'ostaj'),
            'ostajte' => 'ostaj',
            'ostajmo' => 'ostaj',
            'ostajao' => 'ostaj',
            'ostajem' => 'ostaj',
            'lučujte' => 'luč',
            'lučujmo' => 'luč',
            'ostaješ' => 'ostaj',
            'ostajat' => 'ostaj',
            'lučiste' => 'luč',
            'lučismo' => 'luč',
            'lučuješ' => 'luč',
            'lučujem' => 'luč',
            'ganijom' => 'ganij',
            'ostajah' => 'ostaj',
            'anjijih' => 'anj',
            'istadoh' => 'ista',
            'šnjijim' => 'šnj',
            'ovitije' => 'ovit',
            'astijih' => 'ast',
            'evitemu' => 'evit',
            'evitega' => 'evit',
            'astijim' => 'ast',
            'enjijim' => 'enj',
            'enjijih' => 'enj',
            'astijem' => 'ast',
            'rošivši' => 'roši',
            'rošismo' => 'roši',
            'rošiste' => 'roši',
            'ovitemu' => 'ovit',
            'skijega' => 'sk',
            'ovitiji' => 'ovit',
            'ovitija' => 'ovit',
            'astijeg' => 'ast',
            'ovitoga' => 'ovit',
            'šnjijoj' => 'šnj',
            'šnjijih' => 'šnj',
            'ugovima' => 'ugov',
            'urajuci' => array('no_diacritics' => 'ur'),
            'ošijemu' => 'oš',
            'ovitome' => 'ovit',
            'ošijima' => 'oš',
            'ovitomu' => 'ovit',
            'ovitima' => 'ovit',
            'ošijega' => 'oš',
            'irujuci' => array('no_diacritics' => 'ir'),
            'šnjijeg' => 'šnj',
            'šnjijem' => 'šnj',
            'astijoj' => 'ast',
            'enjijoj' => 'enj',
            'enjijem' => 'enj',
            'evitija' => 'evit',
            'onovima' => 'on',
            'osijemu' => array('no_diacritics' => 'os'),
            'osijega' => array('no_diacritics' => 'os'),
            'evitima' => 'evit',
            'osijima' => array('no_diacritics' => 'os'),
            'rosiste' => array('no_diacritics' => 'rosi'),
            'rosivsi' => array('no_diacritics' => 'rosi'),
            'rosismo' => array('no_diacritics' => 'rosi'),
            'snjijem' => array('no_diacritics' => 'os'),
            'evitomu' => 'evit',
            'ovitega' => 'ovit',
            'evitoga' => 'evit',
            'evitome' => 'evit',
            'evitiji' => 'evit',
            'enjijeg' => 'enj',
            'irajuci' => array('no_diacritics' => 'ir'),
            'avajući' => 'ava',
            'rarijem' => 'rari',
            'evajući' => 'eva',
            'evitije' => 'evit',
            'skijemu' => 'sk',
            'činome' => 'čin',
            'ošijim' => 'oš',
            'činomu' => 'čin',
            'činoga' => 'čin',
            'činima' => 'čin',
            'ostaje' => 'ostaj',
            'ošijeg' => 'oš',
            'ošijem' => 'oš',
            'estima' => 'est',
            'ovitoj' => 'ovit',
            'rošila' => 'roši',
            'astije' => 'ast',
            'štvima' => 'štv',
            'latima' => 'lat',
            'ostaju' => 'ostaj',
            'stvima' => 'stv',
            'astemu' => 'ast',
            'astega' => 'ast',
            'ikatom' => 'ikat',
            'istaju' => 'istaj',
            'rošiše' => 'roši',
            'rošile' => 'roši',
            'rošimo' => 'roši',
            'rošite' => 'roši',
            'rošiti' => 'roši',
            'rošili' => 'roši',
            'istaje' => 'istaj',
            'astiji' => 'ast',
            'rošilo' => 'roši',
            'anjima' => 'anj',
            'tanije' => 'tanij',
            'ošijih' => 'oš',
            'skijim' => 'sk',
            'astalo' => 'asta',
            'astale' => 'asta',
            'jašima' => 'jaš',
            'jašama' => 'jaš',
            'skijih' => 'sk',
            'manija' => 'manij',
            'maniju' => 'manij',
            'našima' => 'naš',
            'evitem' => 'evit',
            'našama' => 'naš',
            'astala' => 'asta',
            'manije' => 'manij',
            'maniji' => 'manij',
            'evitog' => 'evit',
            'evitom' => 'evit',
            'evitim' => 'evit',
            'tašima' => 'taš',
            'tašama' => 'taš',
            'evitih' => 'evit',
            'evitoj' => 'evit',
            'skijem' => 'sk',
            'astali' => 'asta',
            'astadu' => 'asta',
            'ošijoj' => 'oš',
            'astija' => 'ast',
            'ovitih' => 'ovit',
            'ovitim' => 'ovit',
            'ovitom' => 'ovit',
            'vašima' => 'vaš',
            'vašama' => 'vaš',
            'ovitog' => 'ovit',
            'tanija' => 'tanij',
            'taniju' => 'tanij',
            'gašima' => 'gaš',
            'gašama' => 'gaš',
            'taniji' => 'tanij',
            'kašama' => 'kaš',
            'bašima' => 'baš',
            'bašama' => 'baš',
            'ovitem' => 'ovit',
            'skijeg' => 'sk',
            'astade' => 'asta',
            'astati' => 'asta',
            'skijoj' => 'sk',
            'astane' => 'asta',
            'astanu' => 'asta',
            'kašima' => 'kaš',
            'činama' => 'čin',
            'kstima' => 'kst',
            'estama' => 'est',
            'enjima' => 'enj',
            'enjoga' => 'enj',
            'ovijoj' => 'ov',
            'ovijih' => 'ov',
            'ovijim' => 'ov',
            'irivat' => 'ir',
            'irujem' => 'ir',
            'iruješ' => 'ir',
            'ovijem' => 'ov',
            'ovijeg' => 'ov',
            'enjome' => 'enj',
            'enjomu' => 'enj',
            'irujmo' => 'ir',
            'movima' => 'mov',
            'irujte' => 'ir',
            'istade' => 'ista',
            'irasmo' => 'ir',
            'iraste' => 'ir',
            'ačismo' => 'ač',
            'ačiste' => 'ač',
            'ivijoj' => 'iv',
            'ivijih' => 'iv',
            'ivijim' => 'iv',
            'ivijem' => 'iv',
            'ivijeg' => 'iv',
            'enjiji' => 'enj',
            'enjega' => 'enj',
            'uvajte' => 'uva',
            'enjija' => 'enj',
            'ivajmo' => 'iva',
            'anjome' => 'anj',
            'anjoga' => 'anj',
            'anjiji' => 'anj',
            'anjije' => 'anj',
            'anjija' => 'anj',
            'anjega' => 'anj',
            'anjemu' => 'anj',
            'anjiju' => 'anj',
            'avajte' => 'ava',
            'etavsi' => array('no_diacritics' => 'et'),
            'ivasmo' => 'iva',
            'ivaste' => 'iva',
            'ivajte' => 'iva',
            'uvajmo' => 'uva',
            'elijoj' => 'el',
            'avajmo' => 'ava',
            'elijih' => 'el',
            'elijim' => 'el',
            'elijem' => 'el',
            'elijeg' => 'el',
            'avaste' => 'ava',
            'avasmo' => 'ava',
            'lovima' => 'lov',
            'enjemu' => 'enj',
            'uvasmo' => 'uva',
            'uvaste' => 'uva',
            'enjije' => 'enj',
            'ačavši' => 'ača',
            'astoga' => 'ast',
            'evaste' => 'eva',
            'škijim' => 'šk',
            'škijih' => 'šk',
            'škijoj' => 'šk',
            'škijeg' => 'šk',
            'šnjima' => 'šnj',
            'šnjiji' => 'šnj',
            'šnjije' => 'šnj',
            'šnjija' => 'šnj',
            'šnjiju' => 'šnj',
            'ostama' => 'ost',
            'ostima' => 'ost',
            'evajmo' => 'eva',
            'evasmo' => 'eva',
            'jetise' => array('no_diacritics' => 'jeti'),
            'kijega' => 'k',
            'astaje' => 'astaj',
            'kstama' => 'kst',
            'anjomu' => 'anj',
            'astaju' => 'astaj',
            'kijima' => 'k',
            'kijemu' => 'k',
            'istama' => 'ist',
            'istima' => 'ist',
            'astima' => 'ast',
            'astomu' => 'ast',
            'astome' => 'ast',
            'ištima' => 'išt',
            'injase' => array('no_diacritics' => 'inj'),
            'ačasmo' => 'ača',
            'urasmo' => 'ur',
            'ačaste' => 'ača',
            'evijoj' => 'ev',
            'evijih' => 'ev',
            'enjiju' => 'enj',
            'evijim' => 'ev',
            'evijem' => 'ev',
            'irujes' => array('no_diacritics' => 'ir'),
            'iravsi' => array('no_diacritics' => 'ir'),
            'evijeg' => 'ev',
            'šnjemu' => 'šnj',
            'šnjega' => 'šnj',
            'evajte' => 'eva',
            'uraste' => 'ur',
            'škijem' => 'šk',
            'urajmo' => 'ur',
            'urajte' => 'ur',
            'avijoj' => 'av',
            'avijih' => 'av',
            'avijim' => 'av',
            'avijem' => 'av',
            'avijeg' => 'av',
            'šnjoga' => 'šnj',
            'šnjome' => 'šnj',
            'šnjomu' => 'šnj',
            'ovaste' => 'ova',
            'ovasmo' => 'ova',
            'iravši' => 'ir',
            'istane' => 'ista',
            'istati' => 'ista',
            'luceno' => array('no_diacritics' => 'luc'),
            'lucile' => array('no_diacritics' => 'luc'),
            'lucili' => array('no_diacritics' => 'luc'),
            'lucilo' => array('no_diacritics' => 'luc'),
            'lajima' => 'laj',
            'lajama' => 'laj',
            'lajeve' => 'laj',
            'lajeva' => 'laj',
            'lajevi' => 'laj',
            'aninom' => 'an',
            'lucena' => array('no_diacritics' => 'luc'),
            'luceni' => array('no_diacritics' => 'luc'),
            'lucene' => array('no_diacritics' => 'luc'),
            'lucimo' => array('no_diacritics' => 'luc'),
            'lucuju' => array('no_diacritics' => 'luc'),
            'đajima' => 'đaj',
            'đajama' => 'đaj',
            'panija' => 'panij',
            'đajeva' => 'đaj',
            'đajevi' => 'đaj',
            'ćajima' => 'ćaj',
            'ćajama' => 'ćaj',
            'ćajeve' => 'ćaj',
            'ćajeva' => 'ćaj',
            'ćajevi' => 'ćaj',
            'čajima' => 'čaj',
            'čajama' => 'čaj',
            'čajeve' => 'čaj',
            'lucila' => array('no_diacritics' => 'luc'),
            'ganiji' => 'ganij',
            'čajevi' => 'čaj',
            'ganija' => 'ganij',
            'atijeg' => 'at',
            'atijem' => 'at',
            'mijama' => 'mij',
            'mijima' => 'mij',
            'nijama' => 'nij',
            'atijih' => 'at',
            'atijim' => 'at',
            'lijama' => 'lij',
            'lijima' => 'lij',
            'dijama' => 'dij',
            'dijima' => 'dij',
            'etavši' => 'et',
            'ganiju' => 'ganij',
            'rajevi' => 'raj',
            'cijama' => 'cij',
            'cijima' => 'cij',
            'ganije' => 'ganij',
            'bijama' => 'bij',
            'bijima' => 'bij',
            'luciti' => array('no_diacritics' => 'luc'),
            'lucite' => array('no_diacritics' => 'luc'),
            'lucise' => array('no_diacritics' => 'luc'),
            'lucuje' => array('no_diacritics' => 'luc'),
            'rajima' => 'raj',
            'rajama' => 'raj',
            'rajeve' => 'raj',
            'rajeva' => 'raj',
            'čajeva' => 'čaj',
            'snjiji' => array('no_diacritics' => 'snj'),
            'žijama' => 'žij',
            'kasama' => array('no_diacritics' => 'kas'),
            'rosila' => array('no_diacritics' => 'rosi'),
            'acavsi' => array('no_diacritics' => 'aca'),
            'acaste' => array('no_diacritics' => 'aca'),
            'acasmo' => array('no_diacritics' => 'aca'),
            'jasima' => array('no_diacritics' => 'jas'),
            'jasama' => array('no_diacritics' => 'jas'),
            'tasima' => array('no_diacritics' => 'tas'),
            'tasama' => array('no_diacritics' => 'tas'),
            'gasima' => array('no_diacritics' => 'gas'),
            'gasama' => array('no_diacritics' => 'gas'),
            'nasama' => array('no_diacritics' => 'nas'),
            'nasima' => array('no_diacritics' => 'nas'),
            'kasima' => array('no_diacritics' => 'kas'),
            'rosimo' => array('no_diacritics' => 'rosi'),
            'nijemu' => 'n',
            'nijega' => 'n',
            'nijima' => 'n',
            'vasama' => array('no_diacritics' => 'vas'),
            'vasima' => array('no_diacritics' => 'vas'),
            'basama' => array('no_diacritics' => 'bas'),
            'basima' => array('no_diacritics' => 'bas'),
            'astuci' => array('no_diacritics' => 'as'),
            'cinima' => array('no_diacritics' => 'cin'),
            'cinome' => array('no_diacritics' => 'cin'),
            'cinama' => array('no_diacritics' => 'cin'),
            'cinomu' => array('no_diacritics' => 'cin'),
            'cinoga' => array('no_diacritics' => 'cin'),
            'rosile' => array('no_diacritics' => 'rosi'),
            'rosilo' => array('no_diacritics' => 'rosi'),
            'cajima' => 'caj',
            'otijem' => 'oti',
            'cajama' => 'caj',
            'cajeve' => 'caj',
            'cajeva' => 'caj',
            'cajevi' => 'caj',
            'snjija' => array('no_diacritics' => 'snj'),
            'snjije' => array('no_diacritics' => 'snj'),
            'snjiju' => array('no_diacritics' => 'snj'),
            'snjima' => array('no_diacritics' => 'snj'),
            'snjemu' => array('no_diacritics' => 'snj'),
            'snjomu' => array('no_diacritics' => 'snj'),
            'snjome' => array('no_diacritics' => 'snj'),
            'snjega' => array('no_diacritics' => 'snj'),
            'snjoga' => array('no_diacritics' => 'snj'),
            'rariju' => 'rari',
            'rosite' => array('no_diacritics' => 'rosi'),
            'rarija' => 'rari',
            'osijih' => array('no_diacritics' => 'os'),
            'osijim' => array('no_diacritics' => 'os'),
            'osijem' => array('no_diacritics' => 'os'),
            'osijeg' => array('no_diacritics' => 'os'),
            'osijoj' => array('no_diacritics' => 'os'),
            'govima' => 'g',
            'logama' => 'log',
            'acismo' => array('no_diacritics' => 'ac'),
            'aciste' => array('no_diacritics' => 'ac'),
            'rositi' => array('no_diacritics' => 'rosi'),
            'rosili' => array('no_diacritics' => 'rosi'),
            'rosise' => array('no_diacritics' => 'rosi'),
            'žijima' => 'žij',
            'đajeve' => 'đaj',
            'jetiše' => 'jeti',
            'lučilo' => 'luč',
            'fijama' => 'fij',
            'astite' => 'as',
            'fijima' => 'fij',
            'ijalom' => 'ijal',
            'injati' => 'inj',
            'sijama' => 'sij',
            'ostali' => 'osta',
            'lučila' => 'luč',
            'astući' => 'as',
            'lučite' => 'luč',
            'ostala' => 'osta',
            'ostadu' => 'osta',
            'injemo' => 'inj',
            'lučimo' => 'luč',
            'ramama' => 'ram',
            'ostanu' => 'osta',
            'ostane' => 'osta',
            'jetite' => 'jeti',
            'sijima' => 'sij',
            'jetili' => 'jeti',
            'ostalo' => 'osta',
            'rijima' => 'rij',
            'rijama' => 'rij',
            'lučene' => 'luč',
            'lučeni' => 'luč',
            'lučeno' => 'luč',
            'ostale' => 'osta',
            'nalama' => 'nal',
            'nalima' => 'nal',
            'lučili' => 'luč',
            'zijama' => 'zij',
            'lemima' => 'lem',
            'lemama' => 'lem',
            'pijama' => 'pij',
            'pijima' => 'pij',
            'lučena' => 'luč',
            'jetimo' => 'jeti',
            'lučile' => 'luč',
            'tijima' => 'tij',
            'jetiti' => 'jeti',
            'tijama' => 'tij',
            'istadu' => 'ista',
            'paniju' => 'panij',
            'injalo' => 'inj',
            'jetilo' => 'jeti',
            'zijima' => 'zij',
            'ostati' => 'osta',
            'istanu' => 'ista',
            'istala' => 'ista',
            'istale' => 'ista',
            'istali' => 'ista',
            'gijima' => 'gij',
            'gijama' => 'gij',
            'istalo' => 'ista',
            'injale' => 'inj',
            'jetile' => 'jeti',
            'injaše' => 'inj',
            'ozilom' => 'ozil',
            'injala' => 'inj',
            'injali' => 'inj',
            'raniji' => 'ranij',
            'ostade' => 'osta',
            'lučiti' => 'luč',
            'jetila' => 'jeti',
            'lučiše' => 'luč',
            'astimo' => 'as',
            'astete' => 'as',
            'lučuju' => 'luč',
            'astemo' => 'as',
            'panije' => 'panij',
            'ranije' => 'ranij',
            'raniju' => 'ranij',
            'ranija' => 'ranij',
            'paniji' => 'panij',
            'lučuje' => 'luč',
            'injete' => 'inj',
            'injahu' => 'inj',
            'kijem' => 'k',
            'acene' => array('no_diacritics' => 'ac'),
            'kijih' => 'k',
            'aceci' => array('no_diacritics' => 'ac'),
            'kijim' => 'k',
            'aceni' => array('no_diacritics' => 'ac'),
            'acemu' => array('no_diacritics' => 'ac'),
            'acena' => array('no_diacritics' => 'ac'),
            'aceno' => array('no_diacritics' => 'ac'),
            'acoga' => array('no_diacritics' => 'ac'),
            'ecome' => array('no_diacritics' => 'ec'),
            'ecoga' => array('no_diacritics' => 'ec'),
            'kijoj' => 'k',
            'acega' => array('no_diacritics' => 'ac'),
            'elama' => 'e',
            'acilo' => array('no_diacritics' => 'ac'),
            'osomu' => array('no_diacritics' => 'os'),
            'osiju' => array('no_diacritics' => 'os'),
            'osiji' => array('no_diacritics' => 'os'),
            'osije' => array('no_diacritics' => 'os'),
            'osija' => array('no_diacritics' => 'os'),
            'osoga' => array('no_diacritics' => 'os'),
            'osega' => array('no_diacritics' => 'os'),
            'osome' => array('no_diacritics' => 'os'),
            'osemu' => array('no_diacritics' => 'os'),
            'evano' => 'eva',
            'evati' => 'eva',
            'evaše' => 'eva',
            'osima' => array('no_diacritics' => 'os'),
            'isama' => array('no_diacritics' => 'is'),
            'etuci' => array('no_diacritics' => 'et'),
            'evahu' => 'eva',
            'injem' => 'inj',
            'evate' => 'eva',
            'ecemu' => array('no_diacritics' => 'ec'),
            'acili' => array('no_diacritics' => 'ac'),
            'acome' => array('no_diacritics' => 'ac'),
            'acile' => array('no_diacritics' => 'ac'),
            'acila' => array('no_diacritics' => 'ac'),
            'acise' => array('no_diacritics' => 'ac'),
            'aciti' => array('no_diacritics' => 'ac'),
            'evaju' => 'eva',
            'acite' => array('no_diacritics' => 'ac'),
            'acomu' => array('no_diacritics' => 'ac'),
            'acimo' => array('no_diacritics' => 'ac'),
            'evani' => 'eva',
            'acima' => array('no_diacritics' => 'ac'),
            'evala' => 'eva',
            'evale' => 'eva',
            'evali' => 'eva',
            'evalo' => 'eva',
            'evamo' => 'eva',
            'evana' => 'eva',
            'evane' => 'eva',
            'kijeg' => 'k',
            'ucemu' => array('no_diacritics' => 'uc'),
            'ecima' => array('no_diacritics' => 'ec'),
            'astao' => 'asta',
            'gasom' => array('no_diacritics' => 'gas'),
            'gasem' => array('no_diacritics' => 'gas'),
            'tasom' => array('no_diacritics' => 'tas'),
            'tasem' => array('no_diacritics' => 'tas'),
            'jasom' => array('no_diacritics' => 'jas'),
            'jasem' => array('no_diacritics' => 'jas'),
            'acala' => array('no_diacritics' => 'aca'),
            'nivsi' => array('no_diacritics' => 'ni'),
            'oviti' => 'ovit',
            'ovite' => 'ovit',
            'ovito' => 'ovit',
            'acana' => array('no_diacritics' => 'aca'),
            'acalo' => array('no_diacritics' => 'aca'),
            'acamo' => array('no_diacritics' => 'aca'),
            'acano' => array('no_diacritics' => 'aca'),
            'ujuci' => array('no_diacritics' => 'uj'),
            'snega' => array('no_diacritics' => 'sn'),
            'ovita' => 'ovit',
            'vasem' => array('no_diacritics' => 'vas'),
            'cinom' => array('no_diacritics' => 'cin'),
            'astes' => array('no_diacritics' => 'as'),
            'basem' => array('no_diacritics' => 'bas'),
            'basom' => array('no_diacritics' => 'bas'),
            'istat' => 'ista',
            'istao' => 'ista',
            'vasom' => array('no_diacritics' => 'vas'),
            'nasem' => array('no_diacritics' => 'nas'),
            'nuvsi' => array('no_diacritics' => 'n'),
            'ajuci' => array('no_diacritics' => 'a'),
            'tavsi' => array('no_diacritics' => 'ta'),
            'kasem' => array('no_diacritics' => 'kas'),
            'kasom' => array('no_diacritics' => 'kas'),
            'snemu' => array('no_diacritics' => 'sn'),
            'nasom' => array('no_diacritics' => 'nas'),
            'acase' => array('no_diacritics' => 'aca'),
            'ovitu' => 'ovit',
            'ecega' => array('no_diacritics' => 'ec'),
            'ucima' => array('no_diacritics' => 'uc'),
            'tajmo' => 'ta',
            'tajte' => 'ta',
            'tavši' => 'ta',
            'avaju' => 'ava',
            'ucega' => array('no_diacritics' => 'uc'),
            'ucoga' => array('no_diacritics' => 'uc'),
            'ucome' => array('no_diacritics' => 'uc'),
            'tasmo' => 'ta',
            'ucomu' => array('no_diacritics' => 'uc'),
            'ecomu' => array('no_diacritics' => 'ec'),
            'astog' => 'ast',
            'astom' => 'ast',
            'astim' => 'ast',
            'astih' => 'ast',
            'astoj' => 'ast',
            'taste' => 'ta',
            'ostao' => 'osta',
            'urase' => array('no_diacritics' => 'ur'),
            'acane' => array('no_diacritics' => 'aca'),
            'acahu' => array('no_diacritics' => 'aca'),
            'irase' => array('no_diacritics' => 'ir'),
            'acale' => array('no_diacritics' => 'aca'),
            'acati' => array('no_diacritics' => 'aca'),
            'acani' => array('no_diacritics' => 'aca'),
            'acali' => array('no_diacritics' => 'aca'),
            'acate' => array('no_diacritics' => 'aca'),
            'acaju' => array('no_diacritics' => 'aca'),
            'ostat' => 'osta',
            'astat' => 'asta',
            'rosih' => array('no_diacritics' => 'rosi'),
            'rosim' => array('no_diacritics' => 'rosi'),
            'rosio' => array('no_diacritics' => 'rosi'),
            'rosis' => array('no_diacritics' => 'rosi'),
            'astem' => 'ast',
            'rosit' => array('no_diacritics' => 'rosi'),
            'snjoj' => array('no_diacritics' => 'snj'),
            'avana' => 'ava',
            'avala' => 'ava',
            'ačaju' => 'ača',
            'ačate' => 'ača',
            'ačaše' => 'ača',
            'ačala' => 'ača',
            'ačalo' => 'ača',
            'lučih' => 'luč',
            'ačali' => 'ača',
            'ačale' => 'ača',
            'ačana' => 'ača',
            'ačati' => 'ača',
            'avase' => array('no_diacritics' => 'ava'),
            'evase' => array('no_diacritics' => 'eva'),
            'ačano' => 'ača',
            'ačani' => 'ača',
            'ačane' => 'ača',
            'nuvši' => 'n',
            'nusmo' => 'n',
            'nuste' => 'n',
            'ačamo' => 'ača',
            'ačahu' => 'ača',
            'nismo' => 'ni',
            'lučiš' => 'luč',
            'ačimo' => 'ač',
            'ačite' => 'ač',
            'ačiše' => 'ač',
            'lučuj' => 'luč',
            'lučio' => 'luč',
            'lučen' => 'luč',
            'lučim' => 'luč',
            'ačeći' => 'ač',
            'lučit' => 'luč',
            'ačila' => 'ač',
            'ačilo' => 'ač',
            'ačili' => 'ač',
            'ačile' => 'ač',
            'ačena' => 'ač',
            'ačeno' => 'ač',
            'ačeni' => 'ač',
            'ačene' => 'ač',
            'nivši' => 'ni',
            'niste' => 'ni',
            'iruju' => 'ir',
            'ovase' => array('no_diacritics' => 'ova'),
            'urano' => 'ur',
            'urani' => 'ur',
            'urane' => 'ur',
            'injes' => array('no_diacritics' => 'inj'),
            'uvase' => array('no_diacritics' => 'uva'),
            'atoga' => 'at',
            'atima' => 'at',
            'jetim' => 'jeti',
            'uralo' => 'ur',
            'jetis' => array('no_diacritics' => 'jeti'),
            'atemu' => 'at',
            'atega' => 'at',
            'jetiš' => 'jeti',
            'jetit' => 'jeti',
            'ilima' => 'i',
            'jetih' => 'jeti',
            'jetio' => 'jeti',
            'urana' => 'ur',
            'urale' => 'ur',
            'ajući' => 'a',
            'ivase' => array('no_diacritics' => 'iva'),
            'etimo' => 'et',
            'etemo' => 'et',
            'etući' => 'et',
            'uramo' => 'ur',
            'urate' => 'ur',
            'uraju' => 'ur',
            'urati' => 'ur',
            'atiju' => 'at',
            'atome' => 'at',
            'uraše' => 'ur',
            'urahu' => 'ur',
            'urala' => 'ur',
            'urali' => 'ur',
            'atija' => 'at',
            'atije' => 'at',
            'atiji' => 'at',
            'atomu' => 'at',
            'ačiti' => 'ač',
            'iruje' => 'ir',
            'avale' => 'ava',
            'ivala' => 'iva',
            'avate' => 'ava',
            'avati' => 'ava',
            'avaše' => 'ava',
            'avahu' => 'ava',
            'ajima' => 'aj',
            'ajama' => 'aj',
            'ivaju' => 'iva',
            'ivale' => 'iva',
            'avani' => 'ava',
            'ivali' => 'iva',
            'ivalo' => 'iva',
            'ajeve' => 'aj',
            'cinoj' => array('no_diacritics' => 'cin'),
            'ajeva' => 'aj',
            'ivamo' => 'iva',
            'ivana' => 'iva',
            'ivane' => 'iva',
            'avano' => 'ava',
            'avane' => 'ava',
            'ajevi' => 'aj',
            'inješ' => 'inj',
            'avali' => 'ava',
            'snjog' => array('no_diacritics' => 'snj'),
            'snjeg' => array('no_diacritics' => 'snj'),
            'snjom' => array('no_diacritics' => 'snj'),
            'snjem' => array('no_diacritics' => 'snj'),
            'snjim' => array('no_diacritics' => 'snj'),
            'snjih' => array('no_diacritics' => 'snj'),
            'injat' => 'inj',
            'ilama' => 'i',
            'injah' => 'inj',
            'injao' => 'inj',
            'avalo' => 'ava',
            'avamo' => 'ava',
            'cinih' => array('no_diacritics' => 'cin'),
            'cinog' => array('no_diacritics' => 'cin'),
            'esama' => array('no_diacritics' => 'es'),
            'istem' => array('no_diacritics' => 'ist'),
            'ivani' => 'iva',
            'asteš' => 'as',
            'irale' => 'ir',
            'lucim' => array('no_diacritics' => 'luc'),
            'uvate' => 'uva',
            'uvati' => 'uva',
            'uvaše' => 'uva',
            'uvahu' => 'uva',
            'lucit' => array('no_diacritics' => 'luc'),
            'lucih' => array('no_diacritics' => 'luc'),
            'lucis' => array('no_diacritics' => 'luc'),
            'irati' => 'ir',
            'uvani' => 'uva',
            'iramo' => 'ir',
            'irate' => 'ir',
            'iraju' => 'ir',
            'iraše' => 'ir',
            'irahu' => 'ir',
            'irala' => 'ir',
            'iralo' => 'ir',
            'irali' => 'ir',
            'uvano' => 'uva',
            'uvane' => 'uva',
            'alima' => 'a',
            'lucuj' => array('no_diacritics' => 'luc'),
            'alama' => 'a',
            'lucen' => array('no_diacritics' => 'luc'),
            'ivano' => 'iva',
            'ivate' => 'iva',
            'ivati' => 'iva',
            'ivaše' => 'iva',
            'ivahu' => 'iva',
            'ijemo' => 'i',
            'uvana' => 'uva',
            'ijete' => 'i',
            'lucio' => array('no_diacritics' => 'luc'),
            'uvaju' => 'uva',
            'uvala' => 'uva',
            'uvale' => 'uva',
            'uvali' => 'uva',
            'uvalo' => 'uva',
            'uvamo' => 'uva',
            'cinim' => array('no_diacritics' => 'cin'),
            'isima' => 'is',
            'evitu' => 'evit',
            'olima' => 'ol',
            'logom' => 'log',
            'ugama' => 'ug',
            'ugova' => 'ugov',
            'ugove' => 'ugov',
            'ugovi' => 'ugov',
            'ućima' => 'uć',
            'ućemu' => 'uć',
            'ućega' => 'uć',
            'otija' => 'oti',
            'ućomu' => 'uć',
            'ućome' => 'uć',
            'evemu' => 'ev',
            'evega' => 'ev',
            'ućoga' => 'uć',
            'evoga' => 'ev',
            'evome' => 'ev',
            'rario' => 'rari',
            'otiju' => 'oti',
            'evima' => 'ev',
            'avama' => 'av',
            'cajem' => 'caj',
            'avemu' => 'av',
            'avega' => 'av',
            'avoga' => 'av',
            'avome' => 'av',
            'avomu' => 'av',
            'avima' => 'av',
            'aviji' => 'av',
            'sijem' => 'si',
            'avije' => 'av',
            'avija' => 'av',
            'aviju' => 'av',
            'ujete' => 'uj',
            'ujemo' => 'uj',
            'ujući' => 'uj',
            'lijem' => 'li',
            'evomu' => 'ev',
            'evama' => 'ev',
            'ovale' => 'ova',
            'aćega' => 'ać',
            'ovemu' => 'ov',
            'ovega' => 'ov',
            'aćemu' => 'ać',
            'ovoga' => 'ov',
            'ovome' => 'ov',
            'ovomu' => 'ov',
            'ovima' => 'ov',
            'aćomu' => 'ać',
            'ećoga' => 'eć',
            'aćome' => 'ać',
            'aćoga' => 'ać',
            'oviji' => 'ov',
            'ovije' => 'ov',
            'ovija' => 'ov',
            'oviju' => 'ov',
            'movom' => 'mov',
            'aćima' => 'ać',
            'iviju' => 'iv',
            'eviji' => 'ev',
            'ećima' => 'eć',
            'evije' => 'ev',
            'evija' => 'ev',
            'eviju' => 'ev',
            'ivemu' => 'iv',
            'ivega' => 'iv',
            'ivoga' => 'iv',
            'ivome' => 'iv',
            'ećemu' => 'eć',
            'ivija' => 'iv',
            'ećega' => 'eć',
            'ećomu' => 'eć',
            'ećome' => 'eć',
            'ivomu' => 'iv',
            'ivima' => 'iv',
            'ivama' => 'iv',
            'iviji' => 'iv',
            'ivije' => 'iv',
            'ovalo' => 'ova',
            'ovala' => 'ova',
            'nijih' => 'n',
            'esima' => 'es',
            'tijom' => 'tij',
            'erama' => 'er',
            'erima' => 'er',
            'sijom' => 'sij',
            'orama' => 'or',
            'orima' => 'or',
            'rijom' => 'rij',
            'pijom' => 'pij',
            'drama' => 'dr',
            'tašom' => 'taš',
            'tašem' => 'taš',
            'našom' => 'naš',
            'našem' => 'naš',
            'fijom' => 'fij',
            'jašom' => 'jaš',
            'jašem' => 'jaš',
            'drima' => 'dr',
            'zijom' => 'zij',
            'kašom' => 'kaš',
            'ijalo' => 'ijal',
            'olova' => 'olov',
            'olovi' => 'olov',
            'ozili' => 'ozil',
            'lemom' => 'lem',
            'ozilu' => 'ozil',
            'ozile' => 'ozil',
            'ozila' => 'ozil',
            'ijalu' => 'ijal',
            'arima' => 'ar',
            'ijali' => 'ijal',
            'ijale' => 'ijal',
            'ijala' => 'ijal',
            'evita' => 'evit',
            'ramom' => 'ram',
            'nalom' => 'nal',
            'arama' => 'ar',
            'gijom' => 'gij',
            'kašem' => 'kaš',
            'čajem' => 'čaj',
            'lajem' => 'laj',
            'etima' => 'et',
            'cijom' => 'cij',
            'estom' => 'est',
            'bijom' => 'bij',
            'istom' => 'ist',
            'rajem' => 'raj',
            'kstom' => 'kst',
            'ostom' => 'ost',
            'latom' => 'lat',
            'đajem' => 'đaj',
            'ištem' => 'išt',
            'ćajem' => 'ćaj',
            'ovahu' => 'ova',
            'ovati' => 'ova',
            'ovaše' => 'ova',
            'ovali' => 'ova',
            'etama' => 'et',
            'ikato' => 'ikat',
            'bašom' => 'baš',
            'nijom' => 'nij',
            'bašem' => 'baš',
            'žijom' => 'žij',
            'gašom' => 'gaš',
            'gašem' => 'gaš',
            'mijom' => 'mij',
            'vašom' => 'vaš',
            'vašem' => 'vaš',
            'ešima' => 'eš',
            'ikatu' => 'ikat',
            'ešama' => 'eš',
            'lijom' => 'lij',
            'dijom' => 'dij',
            'išima' => 'iš',
            'išama' => 'iš',
            'ikata' => 'ikat',
            'ikate' => 'ikat',
            'ikati' => 'ikat',
            'nijoj' => 'n',
            'olove' => 'olov',
            'nijim' => 'n',
            'anjog' => 'anj',
            'šnemu' => 'šn',
            'šnega' => 'šn',
            'skomu' => 'sk',
            'skome' => 'sk',
            'činog' => 'čin',
            'činom' => 'čin',
            'činim' => 'čin',
            'činih' => 'čin',
            'činoj' => 'čin',
            'inima' => 'in',
            'inama' => 'in',
            'skija' => 'sk',
            'skije' => 'sk',
            'anjom' => 'anj',
            'skima' => 'sk',
            'anjoj' => 'anj',
            'anjih' => 'anj',
            'anjim' => 'anj',
            'anjem' => 'anj',
            'anjeg' => 'anj',
            'onova' => 'on',
            'rošim' => 'roši',
            'rošiš' => 'roši',
            'rošit' => 'roši',
            'roših' => 'roši',
            'rošio' => 'roši',
            'onove' => 'on',
            'onovi' => 'on',
            'enjeg' => 'enj',
            'enega' => 'en',
            'ošega' => 'oš',
            'šnjom' => 'šnj',
            'škoga' => 'šk',
            'šnjog' => 'šnj',
            'škomu' => 'šk',
            'škima' => 'šk',
            'škija' => 'šk',
            'škije' => 'šk',
            'škiji' => 'šk',
            'stvom' => 'stv',
            'škemu' => 'šk',
            'škega' => 'šk',
            'anemu' => 'an',
            'anega' => 'an',
            'štvom' => 'štv',
            'šnjoj' => 'šnj',
            'enemu' => 'en',
            'šnjih' => 'šnj',
            'šnjim' => 'šnj',
            'šnjem' => 'šnj',
            'šnjeg' => 'šnj',
            'anina' => 'an',
            'aninu' => 'an',
            'anine' => 'an',
            'anima' => 'an',
            'enjog' => 'enj',
            'enjom' => 'enj',
            'enjoj' => 'enj',
            'enjih' => 'enj',
            'enjim' => 'enj',
            'enjem' => 'enj',
            'ošemu' => 'oš',
            'skoga' => 'sk',
            'skiji' => 'sk',
            'elomu' => 'el',
            'ošomu' => 'oš',
            'ošima' => 'oš',
            'ošiji' => 'oš',
            'ošije' => 'oš',
            'ošija' => 'oš',
            'eliju' => 'el',
            'elija' => 'el',
            'elije' => 'el',
            'eliji' => 'el',
            'elima' => 'el',
            'elome' => 'el',
            'nijeg' => 'n',
            'eloga' => 'el',
            'ošiju' => 'oš',
            'elega' => 'el',
            'elemu' => 'el',
            'skemu' => 'sk',
            'lovom' => 'lov',
            'skega' => 'sk',
            'eviti' => 'evit',
            'evite' => 'evit',
            'evito' => 'evit',
            'nijem' => 'n',
            'škome' => 'šk',
            'ošome' => 'oš',
            'ošoga' => 'oš',
            'onima' => 'on',
            'asli' => 'as',
            'skem' => 'sk',
            'sneg' => array('no_diacritics' => 'sn'),
            'ciju' => 'cij',
            'cijo' => 'cij',
            'atih' => 'at',
            'atim' => 'at',
            'atom' => 'at',
            'atog' => 'at',
            'nalo' => 'nal',
            'nalu' => 'nal',
            'nali' => 'nal',
            'tijo' => 'tij',
            'riju' => 'rij',
            'aslo' => 'as',
            'asle' => 'as',
            'ivši' => 'i',
            'ieći' => 'i',
            'ismo' => 'i',
            'etes' => array('no_diacritics' => 'et'),
            'skoj' => 'sk',
            'atoj' => 'at',
            'nale' => 'nal',
            'isem' => array('no_diacritics' => 'is'),
            'dija' => 'dij',
            'nala' => 'nal',
            'asla' => 'as',
            'avsi' => array('no_diacritics' => 'a'),
            'esem' => array('no_diacritics' => 'es'),
            'bije' => 'bij',
            'laju' => 'laj',
            'sije' => 'sij',
            'sijo' => 'sij',
            'raja' => 'raj',
            'raje' => 'raj',
            'snem' => array('no_diacritics' => 'sn'),
            'nuci' => array('no_diacritics' => 'n'),
            'raji' => 'raj',
            'raju' => 'raj',
            'ivas' => array('no_diacritics' => 'iva'),
            'tija' => 'tij',
            'bija' => 'bij',
            'biji' => 'bij',
            'diji' => 'dij',
            'biju' => 'bij',
            'bijo' => 'bij',
            'ieci' => array('no_diacritics' => 'i'),
            'uvas' => array('no_diacritics' => 'uva'),
            'ivsi' => array('no_diacritics' => 'i'),
            'cija' => 'cij',
            'eteš' => 'et',
            'etem' => 'et',
            'cije' => 'cij',
            'ijes' => array('no_diacritics' => 'i'),
            'ciji' => 'cij',
            'škem' => 'šk',
            'rijo' => 'rij',
            'dije' => 'dij',
            'skih' => 'sk',
            'diju' => 'dij',
            'laji' => 'laj',
            'žiju' => 'žij',
            'piji' => 'pij',
            'pije' => 'pij',
            'žijo' => 'žij',
            'pija' => 'pij',
            'gija' => 'gij',
            'gije' => 'gij',
            'ijte' => 'i',
            'giji' => 'gij',
            'giju' => 'gij',
            'gijo' => 'gij',
            'uras' => array('no_diacritics' => 'ur'),
            'ujes' => array('no_diacritics' => 'uj'),
            'žije' => 'žij',
            'zijo' => 'zij',
            'ziju' => 'zij',
            'fija' => 'fij',
            'fije' => 'fij',
            'fiji' => 'fij',
            'fiju' => 'fij',
            'ziji' => 'zij',
            'fijo' => 'fij',
            'luče' => 'luč',
            'zije' => 'zij',
            'zija' => 'zij',
            'luči' => 'luč',
            'žiji' => 'žij',
            'žija' => 'žij',
            'dijo' => 'dij',
            'siji' => 'sij',
            'tiju' => 'tij',
            'lije' => 'lij',
            'ijem' => 'i',
            'iješ' => 'i',
            'tije' => 'tij',
            'skim' => 'sk',
            'skom' => 'sk',
            'liji' => 'lij',
            'lijo' => 'lij',
            'atem' => 'at',
            'ateg' => 'at',
            'skog' => 'sk',
            'riji' => 'rij',
            'piju' => 'pij',
            'ijmo' => 'i',
            'rije' => 'rij',
            'mija' => 'mij',
            'mije' => 'mij',
            'miji' => 'mij',
            'iras' => array('no_diacritics' => 'ir'),
            'rija' => 'rij',
            'tiji' => 'tij',
            'miju' => 'mij',
            'ilom' => 'i',
            'mijo' => 'mij',
            'pijo' => 'pij',
            'nijo' => 'nij',
            'čaje' => 'čaj',
            'laje' => 'laj',
            'vasu' => array('no_diacritics' => 'vas'),
            'ucih' => array('no_diacritics' => 'uc'),
            'ucog' => array('no_diacritics' => 'uc'),
            'uceg' => array('no_diacritics' => 'uc'),
            'ucom' => array('no_diacritics' => 'uc'),
            'ucem' => array('no_diacritics' => 'uc'),
            'ucim' => array('no_diacritics' => 'uc'),
            'ucuh' => array('no_diacritics' => 'uc'),
            'vasa' => array('no_diacritics' => 'vas'),
            'ećeg' => 'eć',
            'ucoj' => array('no_diacritics' => 'uc'),
            'ećih' => 'eć',
            'ećoj' => 'eć',
            'vase' => array('no_diacritics' => 'vas'),
            'vasi' => array('no_diacritics' => 'vas'),
            'onom' => 'on',
            'kasa' => array('no_diacritics' => 'kas'),
            'ećom' => 'eć',
            'ećem' => 'eć',
            'kasu' => array('no_diacritics' => 'kas'),
            'basi' => array('no_diacritics' => 'bas'),
            'acuh' => array('no_diacritics' => 'ac'),
            'acis' => array('no_diacritics' => 'ac'),
            'inom' => 'in',
            'basa' => array('no_diacritics' => 'bas'),
            'basu' => array('no_diacritics' => 'bas'),
            'base' => array('no_diacritics' => 'bas'),
            'ecoj' => array('no_diacritics' => 'ec'),
            'ecuh' => array('no_diacritics' => 'ec'),
            'ećog' => 'eć',
            'ecom' => array('no_diacritics' => 'ec'),
            'ecog' => array('no_diacritics' => 'ec'),
            'eceg' => array('no_diacritics' => 'ec'),
            'ecih' => array('no_diacritics' => 'ec'),
            'ecem' => array('no_diacritics' => 'ec'),
            'ecim' => array('no_diacritics' => 'ec'),
            'ećim' => 'eć',
            'ećuh' => 'eć',
            'kase' => array('no_diacritics' => 'kas'),
            'kasi' => array('no_diacritics' => 'kas'),
            'anom' => 'an',
            'jasu' => array('no_diacritics' => 'jas'),
            'acah' => array('no_diacritics' => 'aca'),
            'noga' => 'n',
            'nomu' => 'n',
            'nome' => 'n',
            'jase' => array('no_diacritics' => 'jas'),
            'jasi' => array('no_diacritics' => 'jas'),
            'jasa' => array('no_diacritics' => 'jas'),
            'nima' => 'n',
            'gasu' => array('no_diacritics' => 'gas'),
            'niju' => 'n',
            'nija' => 'n',
            'nije' => 'n',
            'tase' => array('no_diacritics' => 'tas'),
            'tasa' => array('no_diacritics' => 'tas'),
            'tasu' => array('no_diacritics' => 'tas'),
            'tasi' => array('no_diacritics' => 'tas'),
            'niji' => 'n',
            'acat' => array('no_diacritics' => 'aca'),
            'gase' => array('no_diacritics' => 'gas'),
            'nasa' => array('no_diacritics' => 'nas'),
            'aćom' => 'ać',
            'nase' => array('no_diacritics' => 'nas'),
            'nasi' => array('no_diacritics' => 'nas'),
            'aćim' => 'ać',
            'aćuh' => 'ać',
            'aćog' => 'ać',
            'aćem' => 'ać',
            'aćeg' => 'ać',
            'aćih' => 'ać',
            'gasa' => array('no_diacritics' => 'gas'),
            'aćoj' => 'ać',
            'nasu' => array('no_diacritics' => 'nas'),
            'nega' => 'n',
            'acam' => array('no_diacritics' => 'aca'),
            'acan' => array('no_diacritics' => 'aca'),
            'acao' => array('no_diacritics' => 'aca'),
            'acas' => array('no_diacritics' => 'aca'),
            'nemu' => 'n',
            'acim' => array('no_diacritics' => 'ac'),
            'acio' => array('no_diacritics' => 'ac'),
            'laja' => 'laj',
            'uješ' => 'uj',
            'caju' => 'caj',
            'caji' => 'caj',
            'caje' => 'caj',
            'stvo' => 'stv',
            'stva' => 'stv',
            'stvu' => 'stv',
            'caja' => 'caj',
            'štvo' => 'štv',
            'gasi' => array('no_diacritics' => 'gas'),
            'štva' => 'štv',
            'ujem' => 'uj',
            'ujmo' => 'uj',
            'štvu' => 'štv',
            'anin' => 'an',
            'liju' => 'li',
            'lija' => 'li',
            'čaja' => 'čaj',
            'luci' => array('no_diacritics' => 'luc'),
            'sija' => 'si',
            'ćaje' => 'ćaj',
            'evas' => array('no_diacritics' => 'eva'),
            'đaju' => 'đaj',
            'đaji' => 'đaj',
            'đaje' => 'đaj',
            'đaja' => 'đaj',
            'ćaju' => 'ćaj',
            'ćaji' => 'ćaj',
            'ćaja' => 'ćaj',
            'luce' => array('no_diacritics' => 'luc'),
            'avas' => array('no_diacritics' => 'ava'),
            'škog' => 'šk',
            'čaju' => 'čaj',
            'čaji' => 'čaj',
            'škom' => 'šk',
            'škim' => 'šk',
            'ških' => 'šk',
            'škoj' => 'šk',
            'siju' => 'si',
            'snja' => array('no_diacritics' => 'snj'),
            'acit' => array('no_diacritics' => 'ac'),
            'ućih' => 'uć',
            'cinu' => array('no_diacritics' => 'cin'),
            'ućim' => 'uć',
            'ućuh' => 'uć',
            'ućog' => 'uć',
            'ućem' => 'uć',
            'ućeg' => 'uć',
            'ućom' => 'uć',
            'ućoj' => 'uć',
            'cina' => array('no_diacritics' => 'cin'),
            'cini' => array('no_diacritics' => 'cin'),
            'cino' => array('no_diacritics' => 'cin'),
            'acoj' => array('no_diacritics' => 'ac'),
            'acih' => array('no_diacritics' => 'ac'),
            'acem' => array('no_diacritics' => 'ac'),
            'acom' => array('no_diacritics' => 'ac'),
            'acen' => array('no_diacritics' => 'ac'),
            'acog' => array('no_diacritics' => 'ac'),
            'cine' => array('no_diacritics' => 'cin'),
            'ugom' => 'ug',
            'snje' => array('no_diacritics' => 'snj'),
            'gama' => 'g',
            'snji' => array('no_diacritics' => 'snj'),
            'snjo' => array('no_diacritics' => 'snj'),
            'snju' => array('no_diacritics' => 'snj'),
            'otio' => 'oti',
            'gova' => 'g',
            'gove' => 'g',
            'govi' => 'g',
            'loge' => 'log',
            'osoj' => array('no_diacritics' => 'os'),
            'logu' => 'log',
            'loga' => 'log',
            'osih' => array('no_diacritics' => 'os'),
            'osim' => array('no_diacritics' => 'os'),
            'osem' => array('no_diacritics' => 'os'),
            'osom' => array('no_diacritics' => 'os'),
            'oseg' => array('no_diacritics' => 'os'),
            'osog' => array('no_diacritics' => 'os'),
            'aceg' => array('no_diacritics' => 'ac'),
            'lemu' => 'lem',
            'olom' => 'ol',
            'irao' => 'ir',
            'iruj' => 'ir',
            'iral' => 'ir',
            'iran' => 'ir',
            'iram' => 'ir',
            'iraš' => 'ir',
            'irat' => 'ir',
            'irah' => 'ir',
            'ivem' => 'iv',
            'ivih' => 'iv',
            'iveg' => 'iv',
            'evog' => 'ev',
            'evom' => 'ev',
            'evoj' => 'ev',
            'evih' => 'ev',
            'ačio' => 'ač',
            'ačim' => 'ač',
            'ačiš' => 'ač',
            'ivim' => 'iv',
            'ivoj' => 'iv',
            'ačih' => 'ač',
            'uvah' => 'uva',
            'movu' => 'mov',
            'mova' => 'mov',
            'ovog' => 'ov',
            'ovom' => 'ov',
            'ovoj' => 'ov',
            'ovih' => 'ov',
            'ovim' => 'ov',
            'uvaj' => 'uva',
            'ivom' => 'iv',
            'uvam' => 'uva',
            'uvan' => 'uva',
            'uvao' => 'uva',
            'uvat' => 'uva',
            'uvav' => 'uva',
            'uvaš' => 'uva',
            'oveg' => 'ov',
            'ivog' => 'iv',
            'ačit' => 'ač',
            'ačen' => 'ač',
            'movi' => 'mov',
            'avši' => 'a',
            'avih' => 'av',
            'niti' => 'ni',
            'nila' => 'ni',
            'nilo' => 'ni',
            'nile' => 'ni',
            'nili' => 'ni',
            'avim' => 'av',
            'asmo' => 'a',
            'avom' => 'av',
            'ajmo' => 'a',
            'ajte' => 'a',
            'ajem' => 'a',
            'aloj' => 'a',
            'avem' => 'av',
            'aveg' => 'av',
            'ovao' => 'ova',
            'ovah' => 'ova',
            'avoj' => 'av',
            'nuta' => 'n',
            'evim' => 'ev',
            'avog' => 'av',
            'evem' => 'ev',
            'eveg' => 'ev',
            'ačao' => 'ača',
            'ačam' => 'ača',
            'ačaš' => 'ača',
            'ačat' => 'ača',
            'ačah' => 'ača',
            'ačan' => 'ača',
            'nući' => 'n',
            'nuti' => 'n',
            'nimo' => 'n',
            'nite' => 'n',
            'nemo' => 'n',
            'nete' => 'n',
            'nula' => 'n',
            'nule' => 'n',
            'nuli' => 'n',
            'nuto' => 'n',
            'move' => 'mov',
            'ivaš' => 'iva',
            'ural' => 'ur',
            'šnje' => 'šnj',
            'asta' => 'ast',
            'astu' => 'ast',
            'šnju' => 'šnj',
            'šnjo' => 'šnj',
            'šnji' => 'šnj',
            'kemu' => 'k',
            'kega' => 'k',
            'šnja' => 'šnj',
            'aste' => 'ast',
            'koga' => 'k',
            'kome' => 'k',
            'komu' => 'k',
            'kima' => 'k',
            'kiji' => 'k',
            'kije' => 'k',
            'kija' => 'k',
            'kiju' => 'k',
            'asto' => 'ast',
            'asti' => 'ast',
            'enjo' => 'enj',
            'čine' => 'čin',
            'ošog' => 'oš',
            'ošom' => 'oš',
            'ošoj' => 'oš',
            'oših' => 'oš',
            'ošim' => 'oš',
            'ošem' => 'oš',
            'ošeg' => 'oš',
            'čino' => 'čin',
            'aneg' => 'an',
            'čini' => 'čin',
            'činu' => 'čin',
            'čina' => 'čin',
            'šnem' => 'šn',
            'šneg' => 'šn',
            'enem' => 'en',
            'eneg' => 'en',
            'anem' => 'an',
            'enju' => 'enj',
            'enji' => 'enj',
            'ivav' => 'iva',
            'lovi' => 'lov',
            'avat' => 'ava',
            'avav' => 'ava',
            'avaš' => 'ava',
            'elih' => 'el',
            'elim' => 'el',
            'elem' => 'el',
            'eleg' => 'el',
            'love' => 'lov',
            'avan' => 'ava',
            'lovu' => 'lov',
            'lova' => 'lov',
            'ivah' => 'iva',
            'ivaj' => 'iva',
            'ivam' => 'iva',
            'ivan' => 'iva',
            'ivao' => 'iva',
            'ivat' => 'iva',
            'avao' => 'ava',
            'avam' => 'ava',
            'enje' => 'enj',
            'evav' => 'eva',
            'enja' => 'enj',
            'anju' => 'anj',
            'evah' => 'eva',
            'evaj' => 'eva',
            'evam' => 'eva',
            'evan' => 'eva',
            'evao' => 'eva',
            'evat' => 'eva',
            'evaš' => 'eva',
            'avaj' => 'ava',
            'anjo' => 'anj',
            'anji' => 'anj',
            'anje' => 'anj',
            'anja' => 'anj',
            'elog' => 'el',
            'elom' => 'el',
            'eloj' => 'el',
            'avah' => 'ava',
            'ovat' => 'ova',
            'nulo' => 'n',
            'urat' => 'ur',
            'lati' => 'lat',
            'drom' => 'dr',
            'ešom' => 'eš',
            'ešem' => 'eš',
            'išom' => 'iš',
            'išem' => 'iš',
            'arom' => 'ar',
            'ramo' => 'ram',
            'ramu' => 'ram',
            'rami' => 'ram',
            'rame' => 'ram',
            'rama' => 'ram',
            'lata' => 'lat',
            'late' => 'lat',
            'latu' => 'lat',
            'vaše' => 'vaš',
            'lato' => 'lat',
            'etom' => 'et',
            'lemo' => 'lem',
            'lemi' => 'lem',
            'leme' => 'lem',
            'esta' => 'est',
            'este' => 'est',
            'esti' => 'est',
            'lema' => 'lem',
            'estu' => 'est',
            'esto' => 'est',
            'ista' => 'ist',
            'iste' => 'ist',
            'alom' => 'a',
            'vaši' => 'vaš',
            'tati' => 'ta',
            'kašu' => 'kaš',
            'tašu' => 'taš',
            'taši' => 'taš',
            'taše' => 'taš',
            'naša' => 'naš',
            'našu' => 'naš',
            'naši' => 'naš',
            'naše' => 'naš',
            'jaša' => 'jaš',
            'jašu' => 'jaš',
            'jaši' => 'jaš',
            'jaše' => 'jaš',
            'isom' => 'is',
            'kaša' => 'kaš',
            'kaši' => 'kaš',
            'vašu' => 'vaš',
            'esom' => 'es',
            'kaše' => 'kaš',
            'urah' => 'ur',
            'baša' => 'baš',
            'orom' => 'or',
            'bašu' => 'baš',
            'baši' => 'baš',
            'baše' => 'baš',
            'gaša' => 'gaš',
            'erom' => 'er',
            'gašu' => 'gaš',
            'gaši' => 'gaš',
            'gaše' => 'gaš',
            'vaša' => 'vaš',
            'isti' => 'ist',
            'taša' => 'taš',
            'tali' => 'ta',
            'tano' => 'ta',
            'ostu' => 'ost',
            'osti' => 'ost',
            'oste' => 'ost',
            'tale' => 'ta',
            'osta' => 'ost',
            'tane' => 'ta',
            'ište' => 'išt',
            'ištu' => 'išt',
            'tamo' => 'ta',
            'tani' => 'ta',
            'ksto' => 'kst',
            'osto' => 'ost',
            'ksti' => 'kst',
            'kste' => 'kst',
            'kstu' => 'kst',
            'išta' => 'išt',
            'uran' => 'ur',
            'istu' => 'ist',
            'uraj' => 'ur',
            'tate' => 'ta',
            'ksta' => 'kst',
            'tana' => 'ta',
            'urao' => 'ur',
            'uram' => 'ur',
            'isto' => 'ist',
            'uraš' => 'ur',
            'talo' => 'ta',
            'tala' => 'ta',
            'taju' => 'ta',
            'eri' => 'er',
            'nis' => array('no_diacritics' => 'ni'),
            'ere' => 'er',
            'dri' => 'dr',
            'one' => 'on',
            'ona' => 'on',
            'taj' => 'ta',
            'tan' => 'ta',
            'ela' => 'el',
            'eru' => 'er',
            'eli' => 'el',
            'era' => 'er',
            'kem' => 'k',
            'oru' => 'or',
            'esa' => 'es',
            'oše' => 'oš',
            'ošu' => 'oš',
            'oša' => 'oš',
            'oši' => 'oš',
            'olu' => 'ol',
            'ase' => array('no_diacritics' => 'a'),
            'nem' => 'n',
            'neg' => 'n',
            'ole' => 'ol',
            'esu' => 'es',
            'oli' => 'ol',
            'oni' => 'on',
            'tas' => array('no_diacritics' => 'ta'),
            'tah' => 'ta',
            'ese' => 'es',
            'tat' => 'ta',
            'ori' => 'or',
            'ore' => 'or',
            'taš' => 'ta',
            'tao' => 'ta',
            'ora' => 'or',
            'onu' => 'on',
            'tam' => 'ta',
            'dra' => 'dr',
            'elu' => 'el',
            'kih' => 'k',
            'kog' => 'k',
            'kom' => 'k',
            'ise' => 'is',
            'ška' => 'šk',
            'ško' => 'šk',
            'ške' => 'šk',
            'ske' => 'sk',
            'ški' => 'šk',
            'koj' => 'k',
            'kim' => 'k',
            'ski' => 'sk',
            'esi' => array('no_diacritics' => 'es'),
            'iti' => 'i',
            'sko' => 'sk',
            'ska' => 'sk',
            'sku' => 'sk',
            'ili' => 'i',
            'ila' => 'i',
            'ilo' => 'i',
            'ile' => 'i',
            'kov' => 'k',
            'iju' => 'i',
            'ele' => 'el',
            'ina' => 'in',
            'elo' => 'el',
            'ino' => 'in',
            'dre' => 'dr',
            'keg' => 'k',
            'isi' => array('no_diacritics' => 'is'),
            'dru' => 'dr',
            'inu' => 'in',
            'ini' => 'in',
            'ine' => 'in',
            'ari' => 'ar',
            'ije' => 'i',
            'are' => 'ar',
            'ara' => 'ar',
            'ane' => 'an',
            'ana' => 'an',
            'ani' => 'an',
            'anu' => 'an',
            'aru' => 'ar',
            'imo' => 'i',
            'ite' => 'i',
            'ilu' => 'i',
            'šku' => 'šk',
            'nes' => array('no_diacritics' => 'n'),
            'isa' => 'is',
            'ete' => 'et',
            'uga' => 'ug',
            'nut' => 'n',
            'nuo' => 'n',
            'uća' => 'uć',
            'uće' => 'uć',
            'neš' => 'n',
            'ola' => 'ol',
            'eto' => 'et',
            'etu' => 'et',
            'eti' => 'et',
            'avi' => 'av',
            'ugi' => 'ug',
            'ava' => 'av',
            'eta' => 'et',
            'ace' => array('no_diacritics' => 'ac'),
            'aca' => array('no_diacritics' => 'ac'),
            'isu' => 'is',
            'avu' => 'av',
            'ave' => 'av',
            'avo' => 'av',
            'eća' => 'eć',
            'eće' => 'eć',
            'uge' => 'ug',
            'ugu' => 'ug',
            'iša' => 'iš',
            'aše' => 'a',
            'uju' => 'uj',
            'uje' => 'uj',
            'lio' => 'li',
            'sio' => 'si',
            'ano' => 'a',
            'alo' => 'a',
            'ale' => 'a',
            'ali' => 'a',
            'ala' => 'a',
            'ahu' => 'a',
            'ati' => 'a',
            'osu' => array('no_diacritics' => 'os'),
            'aju' => 'a',
            'aje' => 'a',
            'ate' => 'a',
            'gom' => 'g',
            'amo' => 'a',
            'ugo' => 'ug',
            'nio' => 'ni',
            'niš' => 'ni',
            'osa' => array('no_diacritics' => 'os'),
            'ose' => array('no_diacritics' => 'os'),
            'osi' => array('no_diacritics' => 'os'),
            'išu' => 'iš',
            'aci' => array('no_diacritics' => 'ac'),
            'iva' => 'iv',
            'ivi' => 'iv',
            'ovu' => 'ov',
            'uca' => array('no_diacritics' => 'uc'),
            'uce' => array('no_diacritics' => 'uc'),
            'alu' => 'a',
            'aja' => 'aj',
            'aća' => 'ać',
            'aće' => 'ać',
            'aji' => 'aj',
            'ače' => 'ač',
            'eši' => 'eš',
            'ivu' => 'iv',
            'ive' => 'iv',
            'ivo' => 'iv',
            'ova' => 'ov',
            'ovi' => 'ov',
            'noj' => 'n',
            'emo' => 'e',
            'nom' => 'n',
            'nih' => 'n',
            'nog' => 'n',
            'nim' => 'n',
            'evo' => 'ev',
            'iši' => 'iš',
            'ači' => 'ač',
            'ovo' => 'ov',
            'ove' => 'ov',
            'ato' => 'at',
            'atu' => 'at',
            'ata' => 'at',
            'eca' => array('no_diacritics' => 'ec'),
            'ece' => array('no_diacritics' => 'ec'),
            'iše' => 'iš',
            'ešu' => 'eš',
            'eše' => 'eš',
            'eve' => 'ev',
            'evi' => 'ev',
            'eva' => 'ev',
            'eša' => 'eš',
            'evu' => 'ev',
            'es' => array('no_diacritics' => 'e'),
            'an' => 'a',
            'eš' => 'e',
            'em' => 'e',
            'aš' => 'a',
            'is' => array ( 'no_diacritics' => 'i'),
            'at' => 'a',
            'ah' => 'a',
            'aj' => 'a',
            'am' => 'a',
            'ao' => 'a',
            'as' => array ( 'no_diacritics' => 'a'),
            'ij' => 'i',
            'no' => 'n',
            'im' => 'i',
            'ko' => 'k',
            'ke' => 'k',
            'ku' => 'k',
            'ka' => 'k',
            'ki' => 'k',
            'io' => 'i',
            'ih' => 'i',
            'it' => 'i',
            'iš' => 'i',
            'al' => 'a',
            'ne' => 'n',
            'il' => 'i',
            'ni' => 'n',
            'nu' => 'n',
            'na' => 'n',
            'ga' => 'g',
            'ge' => 'g',
            'gi' => 'g',
            'gu' => 'g',
            'go' => 'g',
            'el' => 'e',
        );

        $replaced = false;

        foreach($strings as $s_key => $s_replace){
            // if this is a string replace and the string key is in the word
            if(is_string($s_replace) && false !== strpos($word, $s_key)){
                $key_len = mb_strlen($s_key);
                $key_pos = mb_strrpos($word, $s_key);

                // see if the key comes at the end of the word by adding the key pos and it's length and seeing if that equals the word length
                if(($key_len + $key_pos) === $length){
                    // if it does, change the ending
                    $word = mb_substr($word, 0, $key_pos) . $s_replace;
                    // say that we've changed the ending
                    $replaced = true;
                    // and exit the loop
                    break;
                }

            }elseif(self::$no_diacritics && false !== strpos($word, $s_key)){
                $key_len = mb_strlen($s_key);
                $key_pos = mb_strrpos($word, $s_key);

                // see if the key comes at the end of the word by adding the key pos and it's length and seeing if that equals the word length
                if(($key_len + $key_pos) === $length){
                    // if it does, change the ending
                    $word = mb_substr($word, 0, $key_pos) . $s_replace['no_diacritics'];                        
                    // say that we've changed the ending
                    $replaced = true;
                    // and exit the loop
                    break;
                }
            }
        }

        return $replaced;
    }

    public static function backwards_step_3($word){

        $r1 = !empty(self::$p1) ? self::$p1: false;

        if(empty($r1)){ // todo I _think_ this is correct, test to make sure
            return false;
        }

        // get the word length
        $length = mb_strlen($word);

        // create the list of word endings to replace
        $strings = array(
            'enom' => '',
            'enoj' => '',
            'enog' => '',
            'enim' => '',
            'enih' => '',
            'anoj' => '',
            'anog' => '',
            'anim' => '',
            'anih' => '',
            'ost' => '',
            'eno' => '',
            'eni' => '',
            'oga' => '',
            'ima' => '',
            'enu' => '',
            'ena' => '',
            'ama' => '',
            'ano' => '',
            'ani' => '',
            'om' => '',
            'og' => '',
            'u' => '',
            'o' => '',
            'i' => '',
            'e' => '',
            'a' => ''
        );

        foreach($strings as $s_key => $s_replace){
            // if this is a string replace and the string key is in the word
            if(false !== strpos($word, $s_key)){
                $key_len = mb_strlen($s_key);
                $key_pos = mb_strrpos($word, $s_key);

                // see if the key comes at the end of the word by adding the key pos and it's length and seeing if that equals the word length
                if(($key_len + $key_pos) === $length){
                    // if it does, remove the ending
                    $word = mb_substr($word, 0, $key_pos);
                    // and exit the loop
                    break;
                }
            }
        }

        return $word;

    }

    public static function Stem($word){
        // convert the cyrillic letters to latin
        $word = self::cyr_to_lat($word);
        // if the wamania project is active, use their strtolower function
        if(class_exists('Wamania\\Snowball\\Utf8')){
            $word = Wamania\Snowball\Utf8::strtolower($word);
        }
        // do some dialect adjusting
        $word = self::prelude($word);
        // mark out the first syllable
        self::mark_regions($word);
        // perform the initial suffix removal
        $word = self::backwards_step_1($word);
        // try removing more suffix
        $removed = self::backwards_step_2($word);
        // if that didn't work, try this removal instead

        if(empty($removed)){
            $word = self::backwards_step_3($word);
        }

        return $word;
    }

}






?>