<?php

/**
 * Work with terms
 */
class Wpil_Term
{
    /**
     * Register services
     */
    public function register()
    {
        foreach (Wpil_Settings::getTermTypes() as $term) {
            add_action($term . '_add_form_fields', [$this, 'showTermSuggestions']);
            add_action($term . '_edit_form', [$this, 'showTermSuggestions']);
            add_action('edited_' . $term, [$this, 'addLinksToTerm']);
            add_action('edited_' . $term, ['Wpil_TargetKeyword', 'update_keywords_on_term_save']);
            add_action($term . '_add_form_fields', [$this, 'showTargetKeywords']);
            add_action($term . '_edit_form', [$this, 'showTargetKeywords']);
        }
    }

    /**
     * Show suggestions on term page
     */
    public static function showTermSuggestions()
    {
        if(empty($_GET['tag_ID']) ||empty($_GET['taxonomy'] || !in_array($_GET['taxonomy'], Wpil_Settings::getTermTypes()))){
            return;
        }

        $term_id = (int)$_GET['tag_ID'];
        $post_id = 0;
        $user = wp_get_current_user();
        $manually_trigger_suggestions = !empty(get_option('wpil_manually_trigger_suggestions', false));
        ?>
        <div id="wpil_link-articles" class="postbox">
            <h2 class="hndle no-drag"><span><?php _e('Link Whisper Suggested Links', 'wpil'); ?></span></h2>
            <div class="inside">
                <?php include WP_INTERNAL_LINKING_PLUGIN_DIR . '/templates/link_list_v2.php';?>
            </div>
        </div>
        <?php
    }

    /**
     * Show target keywords on term page
     */
    public static function showTargetKeywords()
    {
        if(empty($_GET['tag_ID']) ||empty($_GET['taxonomy'] || !in_array($_GET['taxonomy'], Wpil_Settings::getTermTypes()))){
            return;
        }

        $term_id = (int)$_GET['tag_ID'];
        $post_id = 0;
        $user = wp_get_current_user();
        $keywords = Wpil_TargetKeyword::get_keywords_by_post_ids($term_id, 'term');
        $post = new Wpil_Model_Post($term_id, 'term');
        $keyword_sources = Wpil_TargetKeyword::get_active_keyword_sources();
        $is_metabox = true;
        ?>
        <div id="wpil_target-keywords" class="postbox ">
            <h2 class="hndle no-drag"><span><?php _e('Link Whisper Target Keywords', 'wpil'); ?></span></h2>
            <div class="inside"><?php
                include WP_INTERNAL_LINKING_PLUGIN_DIR . '/templates/target_keyword_list.php';
            ?>
            </div>
        </div>
        <?php
    }
    /**
     * Add links to term description on term update
     *
     * @param $term_id
     */
    public static function addLinksToTerm($term_id)
    {
        global $wpdb;

        //get links
        $meta = get_term_meta($term_id,'wpil_links', true);

        if (!empty($meta)) {
            $description = term_description($term_id);

            //add links to the term description
            foreach ($meta as $link) {
                $changed_sentence = Wpil_Post::getSentenceWithAnchor($link);
                $description = preg_replace('/' . preg_quote($link['sentence'], '/') . '/i', $changed_sentence, $description, 1);
            }

            //update term description
            $wpdb->query($wpdb->prepare("UPDATE {$wpdb->prefix}term_taxonomy SET description = %s WHERE term_id = {$term_id} AND description != ''", $description));

            // add links to meta field content areas
            $description .= self::addLinkToMetaContent($term_id);

            //delete links from DB
            delete_term_meta($term_id,'wpil_links');

            if(WPIL_STATUS_LINK_TABLE_EXISTS){
                //update the link table
                $term = new Wpil_Model_Post($term_id, 'term');
                Wpil_Report::update_post_in_link_table($term->setContent($description));
            }
        }

        if (empty(get_option('wpil_post_procession'))) {
            $term = new Wpil_Model_Post($term_id, 'term');
            Wpil_Keyword::addKeywordsToPost($term);
            Wpil_URLChanger::replacePostURLs($term);
        }
    }

    /**
     * Adds links to the term's meta fields
     **/
    public static function addLinkToMetaContent($term_id){
        $meta = get_term_meta($term_id, 'wpil_links', true);

        if (!empty($meta)) {
            $fields = Wpil_Post::getMetaContentFieldList('term');
            if(!empty($fields)){
                foreach($fields as $field){
                    if($content = get_term_meta($term_id, $field, true)){
                        foreach($meta as $link){
                            if(strpos($content, $link['sentence']) !== false){
                                $changed_sentence = Wpil_Post::getSentenceWithAnchor($link);
                                Wpil_Post::insertLink($content, $link['sentence'], $changed_sentence);
                            }
                        }
                        update_term_meta($term_id, $field, $content);
                    }
                }
            }
        }
    }

    /**
     * Get category or tag by slug
     *
     * @param $slug
     * @return WP_Term
     */
    public static function getTermBySlug($slug)
    {
        $term = get_term_by('slug', $slug, 'category');
        if (!$term) {
            $term = get_term_by('slug', $slug, 'post_tag');
        }

        return $term;
    }

    /**
     * Gets all category terms for all active post types
     * 
     * @return array 
     **/
    public static function getAllCategoryTerms(){
        $start = microtime(true);
        $post_types = Wpil_Settings::getPostTypes();
        if(empty($post_types)){
            return false;
        }

        $terms = get_transient('wpil_cached_category_terms');
        if(empty($terms)){

            $skip_terms = array(
                'product_type',
                'product_visibility',
                'product_shipping_class',
            );

            $terms = array();
            foreach($post_types as $type){
                $taxonomies = get_object_taxonomies($type);

                foreach($taxonomies as $taxonomy){
                    if(in_array($taxonomy, $skip_terms)){
                        continue;
                    }

                    $args = array(
                        'taxonomy' => $taxonomy,
                        'hide_empty' => false
                    );
                    $queried_terms = get_terms($args);

                    if(!is_a($terms, 'WP_Error')){
                        foreach($queried_terms as $term){
                            $terms[] = $term;
                        }
                    }
                }
            }

            // sort the terms to find the most used ones
            usort($terms, function($a, $b){
                if($a->count < $b->count){
                    return 1;
                }else if($a->count < $b->count){
                    return -1;
                }else{
                    return 0;
                }
            });

            // _only_ use the top 250 terms to save loading resources for sites that have many, many terms.
            $terms = array_slice($terms, 0, 250);

            // compress the terms to save space
            $terms_to_save = Wpil_Suggestion::compress($terms);

            // cache the terms for 5 minutes
            set_transient('wpil_cached_category_terms', $terms_to_save, MINUTE_IN_SECONDS * 5);
        }else{
            // if there are terms, decompress them
            $terms = Wpil_Suggestion::decompress($terms);
        }

        return $terms;
    }
}
